using System;
using System.Collections;
using System.Data;
using System.Data.SqlTypes;
using System.Reflection;
using gov.va.med.vbecs.ExceptionManagement;
using System.Text.RegularExpressions;
using SPECIMENTESTTHRESHOLD = gov.va.med.vbecs.Common.VbecsTables.SpecimenTestThreshold;

namespace gov.va.med.vbecs.BOL
{

		#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Mike Smith</Developer>
	///	<Developer>Cameron Taylor</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>12/2/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>>Lab test business object layer component.</summary>

		#endregion

	public class LabTest : BaseBusinessObject, IRecordStatusCode
	{

		private int _labTestID;
		private string _LabTestName = "";
		private string _Threshold = "";
		private Common.SpecimenType _SpecimenType;
		private Common.ReportType _ReportType;
		private int _ComponentClassID;
		private System.Guid _SpecimenTestThresholdGuid = Guid.Empty;
		private Common.RecordStatusCode _recordStatusCode;
		private Common.ThresholdType _thresholdType;


		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///
		///	<Case type="0" testid ="264"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid labe test object</ExpectedOutput>
		///	</Case>
		///	
		///	<Case type="1" testid ="354"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Empty constructor
		/// </summary>
		public LabTest()
		{
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2410"> 
		///		<ExpectedInput>DataRow containing LabTest data</ExpectedInput>
		///		<ExpectedOutput>LabTest object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2442"> 
		///		<ExpectedInput>Invalid DataRow</ExpectedInput>
		///		<ExpectedOutput>Invalid LabTest object</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates LabTest object and populates it with data from DataRow
		/// </summary>
		/// <param name="dr"></param>
		public LabTest(DataRow dr)
		{
			this.LoadFromDataRow(dr);
		}

		#region Properties

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6328"> 
		///		<ExpectedInput>RecordStatusCode</ExpectedInput>
		///		<ExpectedOutput>RecordStatusCode</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6329"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// The status of the record
		/// </summary>
		public Common.RecordStatusCode RecordStatusCode
		{
			get
			{
				return _recordStatusCode;
			}
			set
			{
				_recordStatusCode = value;
			}
		}

		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="980"> 
		///		<ExpectedInput>Lab test object.</ExpectedInput>
		///		<ExpectedOutput>Lab test id.</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="994"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Gets or sets the key to the lab test.
		///</summary>
		public System.Guid SpecimenTestThresholdGuid
		{
			get
			{
				return _SpecimenTestThresholdGuid;
			}
			set
			{
				_SpecimenTestThresholdGuid = value;
			}
		}

		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1015"> 
		///		<ExpectedInput>Valid component class id.</ExpectedInput>
		///		<ExpectedOutput>Component class id set to the input string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2026"> 
		///		<ExpectedInput>Int32.MinValue</ExpectedInput>
		///		<ExpectedOutput>A broken rule requiring a component class id.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Gets or sets the component class id.
		///</summary>
		public int ComponentClassId
		{
			get 
			{
				return _ComponentClassID;
			}
			set 
			{
				_ComponentClassID = value;
				RuleBroken(MethodBase.GetCurrentMethod().Name, _ComponentClassID == System.Int32.MinValue);
			}
		}

		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1016"> 
		///		<ExpectedInput>string </ExpectedInput>
		///		<ExpectedOutput>Vista lab test name set to the input string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1017"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Vista lab test name.
		///</summary>
		public string LabTestName
		{
			get 
			{
				return _LabTestName;
			}
			set 
			{
				_LabTestName = value;
				RuleBroken(MethodBase.GetCurrentMethod().Name, _LabTestName.Trim() == String.Empty);
				RuleBroken("SpecimenType", true);
				RuleBroken("Threshold", true);
			}
		}

		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1018"> 
		///		<ExpectedInput>A valid specimen test type.</ExpectedInput>
		///		<ExpectedOutput>Specimen test type set to the input specimen test type.</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1019"> 
		///		<ExpectedInput>An unknown specimen test type.</ExpectedInput>
		///		<ExpectedOutput>A broken rule requiring input of a valid specimen test type.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Type of the specimen.
		///</summary>
		public Common.SpecimenType SpecimenType
		{
			get
			{
				return _SpecimenType;
			}
			set
			{
				if (value != _SpecimenType)
				{
					_SpecimenType = value;
					IsDirty = true;
				}
				RuleBroken(MethodBase.GetCurrentMethod().Name, _SpecimenType == Common.SpecimenType.Unknown);
			}
		}

		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1022"> 
		///		<ExpectedInput>A valid numeric threshold.</ExpectedInput>
		///		<ExpectedOutput>Threshold set to the input threshold.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1021"> 
		///		<ExpectedInput>An invalid text threshold (16 chars).</ExpectedInput>
		///		<ExpectedOutput>Business object exception thrown.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Threshold result of the lab test.
		/// BR_50.05, BR_34.01, BR_34.02
		///</summary>
		public string Threshold
		{
			get
			{
				return _Threshold;
			}
			set
			{
				if (value.Trim().Length > 0)
				{
					Regex regexThreshold = Common.RegularExpressions.Threshold();
					Regex regexTextThreshold = Common.RegularExpressions.ThresholdText();

					if ((this._thresholdType == Common.ThresholdType.Numeric) && !regexThreshold.IsMatch(value.Trim()) ||
						((this._thresholdType == Common.ThresholdType.Text) && !regexTextThreshold.IsMatch(value.Trim())))
					{
						string errorMessage = Common.StrRes.SysErrMsg.Common.InvalidPropertyOrMethod(
												Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name, true),
												value).ResString;
						throw new BOL.BusinessObjectException(errorMessage);
					}
				}

				if (value.Trim() != _Threshold)
				{
					_Threshold = value.Trim();
					IsDirty = true;
				}
				if (this.ThresholdType == Common.ThresholdType.Numeric)
				{
					RuleBroken(MethodBase.GetCurrentMethod().Name, !Common.RegularExpressions.Threshold().IsMatch(_Threshold));
				}
				else if (this.ThresholdType == Common.ThresholdType.Text)
				{
					RuleBroken(MethodBase.GetCurrentMethod().Name, !Common.RegularExpressions.ThresholdText().IsMatch(_Threshold));
				}
				else
				{
					RuleBroken(MethodBase.GetCurrentMethod().Name, false);
				}
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6326"> 
		///		<ExpectedInput>ThresholdType</ExpectedInput>
		///		<ExpectedOutput>ThresholdType</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6327"> 
		///		<ExpectedInput>Invalid ThresholdType for Threshold</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// ThresholdType
		/// </summary>
		public Common.ThresholdType ThresholdType
		{
			get
			{
				return this._thresholdType;
			}
			set
			{
				if (value != _thresholdType)
				{
					_thresholdType = value;
					IsDirty = true;
				}
				if (this.ThresholdType == Common.ThresholdType.Numeric)
				{
					RuleBroken("Threshold", !Common.RegularExpressions.Threshold().IsMatch(_Threshold));
				}
				else if (this.ThresholdType == Common.ThresholdType.Text)
				{
					RuleBroken("Threshold", !Common.RegularExpressions.ThresholdText().IsMatch(_Threshold));
				}
				else
				{
					RuleBroken("Threshold", false);
				}

			}
		}

		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="986"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Report type of the lab test parameter.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1024"> 
		///		<ExpectedInput>Common.ReportType.Unknown</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Report type of the lab test parameter.
		///</summary>
		public Common.ReportType ReportType
		{
			get
			{
				return _ReportType;
			}
			set
			{
				if (value != _ReportType)
				{
					_ReportType = value;
					IsDirty = true;
				}
				RuleBroken(MethodBase.GetCurrentMethod().Name, _ReportType == Common.ReportType.Unknown);			
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6330"> 
		///		<ExpectedInput>int</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6331"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// LabTestId
		/// </summary>
		public int LabTestId
		{
			get
			{
				return this._labTestID;
			}
			set
			{
				this._labTestID = value;
			}
		}

		#endregion

		#region Methods




		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="987"> 
		///		<ExpectedInput>An empty data record.</ExpectedInput>
		///		<ExpectedOutput>Data row equivalent of lab test object.</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="359"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Sets the properties of the lab test object based on supplied data row.
		///</summary>
		///<param name="dtRow">System.Data.DataRow</param>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			dtRow[SPECIMENTESTTHRESHOLD.SpecimenTestThresholdGuid] = SpecimenTestThresholdGuid;
			dtRow[SPECIMENTESTTHRESHOLD.ComponentClassId] = ComponentClassId.Equals(System.Int32.MinValue) ? -1 : ComponentClassId;
			dtRow[SPECIMENTESTTHRESHOLD.VistaLaboratoryTestName] = LabTestName;
			dtRow[SPECIMENTESTTHRESHOLD.SpecimenTypeCode] = Convert.ToChar(SpecimenType);
			dtRow[SPECIMENTESTTHRESHOLD.ThresholdResult] = Threshold.Equals(String.Empty) ? null : Threshold;
			dtRow[SPECIMENTESTTHRESHOLD.ReportTypeId] = (int) ReportType;
			if (dtRow.Table.Columns.Contains(SPECIMENTESTTHRESHOLD.LabTestId))
			{
				dtRow[SPECIMENTESTTHRESHOLD.LabTestId] = this._labTestID;
			}
			dtRow[SPECIMENTESTTHRESHOLD.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			dtRow[SPECIMENTESTTHRESHOLD.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(RecordStatusCode);
            dtRow[SPECIMENTESTTHRESHOLD.LastUpdateUser] = Common.LogonUser.LogonUserName;
			if (dtRow.Table.Columns.Contains(SPECIMENTESTTHRESHOLD.RowVersion))
			{
				if(!dtRow.IsNull(SPECIMENTESTTHRESHOLD.RowVersion))
				{
					dtRow[SPECIMENTESTTHRESHOLD.RowVersion] = RowVersion;
				}
			}
		
			return dtRow;
		}

		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/27/2002</CreationDate>
		///<summary>
		/// Returns a data row containing lab test information.
		///</summary>
		///<param name="dtRow">System.Data.DataRow</param>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			SpecimenTestThresholdGuid = new Guid(dtRow[SPECIMENTESTTHRESHOLD.SpecimenTestThresholdGuid].ToString());
			ComponentClassId = dtRow.IsNull(SPECIMENTESTTHRESHOLD.ComponentClassId) ? System.Int32.MinValue : Convert.ToInt32(dtRow[SPECIMENTESTTHRESHOLD.ComponentClassId].ToString());
			LabTestName = dtRow[SPECIMENTESTTHRESHOLD.VistaLaboratoryTestName].ToString();
			SpecimenType = Common.Utility.GetSpecimenTypeFromString(dtRow[SPECIMENTESTTHRESHOLD.SpecimenTypeCode].ToString());
			if (Common.RegularExpressions.ThresholdText().IsMatch(dtRow[SPECIMENTESTTHRESHOLD.ThresholdResult].ToString()))
			{
				ThresholdType = Common.ThresholdType.Text;
			}
			else if (Common.RegularExpressions.Threshold().IsMatch(dtRow[SPECIMENTESTTHRESHOLD.ThresholdResult].ToString()))
			{
				ThresholdType = Common.ThresholdType.Numeric;
			}
			else
			{
				ThresholdType = Common.ThresholdType.None;
			}
			Threshold = dtRow.IsNull(SPECIMENTESTTHRESHOLD.ThresholdResult) ? "" : dtRow[SPECIMENTESTTHRESHOLD.ThresholdResult].ToString();
			ReportType = dtRow.IsNull(SPECIMENTESTTHRESHOLD.ReportTypeId) ? Common.ReportType.Unknown : (Common.ReportType) Convert.ToInt32(dtRow[SPECIMENTESTTHRESHOLD.ReportTypeId].ToString());
			if (dtRow.Table.Columns.Contains(SPECIMENTESTTHRESHOLD.LabTestId))
			{
				this._labTestID = Convert.ToInt32(dtRow[SPECIMENTESTTHRESHOLD.LabTestId]);
			}
			//DivisionCode = dtRow[SPECIMENTESTTHRESHOLD.DivisionCode].ToString();
			RecordStatusCode = Common.Utility.GetRecordStatusCodeFromString(dtRow[SPECIMENTESTTHRESHOLD.RecordStatusCode].ToString());
			if (dtRow.Table.Columns.Contains(SPECIMENTESTTHRESHOLD.RowVersion))
			{
				if (!dtRow.IsNull(SPECIMENTESTTHRESHOLD.RowVersion))
				{
					RowVersion = (byte[]) dtRow[SPECIMENTESTTHRESHOLD.RowVersion];
				}
			}
			IsDirty = false;
			IsNew = false;
		}

		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="989"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Data table of lab tests for report type and division</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2052"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Calls data access layer to retrieve lab tests of a specific report type within a division.
		///</summary>
		public DataTable GetLabTests()
		{
			return DAL.LabTest.GetLabTests((int) ReportType);
		}

		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="990"> 
		///		<ExpectedInput>Valid data table</ExpectedInput>
		///		<ExpectedOutput>No exception</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2313"> 
		///		<ExpectedInput>Invalid data table</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Calls the data access layer to save the changes in the _LabTests data table.
		///</summary>
		///<param name="labTests">DataTable</param>
		///<param name="lastUpdateFunctionId">UC or calling method</param>
		public void SaveLabTests(DataTable labTests, Common.UpdateFunction lastUpdateFunctionId)
		{
			System.Collections.ArrayList _dtArrayList = new System.Collections.ArrayList();
			System.Collections.ArrayList _sprocArrayList = new System.Collections.ArrayList();
			//
			DataTable labTestChanges = labTests.GetChanges(DataRowState.Modified);
			if (labTestChanges!=null)
			{
				if (labTestChanges.Columns.Contains(Common.VbecsTables.SpecimenType.SpecimenTypeText))
				{
					labTestChanges.Columns.Remove(Common.VbecsTables.SpecimenType.SpecimenTypeText);
				}
				//
				_dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(labTestChanges, lastUpdateFunctionId));
				_sprocArrayList.Add(Common.VbecsStoredProcs.UpdateLabTest.StoredProcName);
			}

			System.Data.DataTable dtInsert = Common.Utility.AppendLastUpdateInformation(labTests.GetChanges(DataRowState.Added), lastUpdateFunctionId);
			if (dtInsert!=null)
			{
				//Logic added here because it was removed from the sproc
				foreach(System.Data.DataRow dr in dtInsert.Rows)
				{
					if (dtInsert.Columns.Contains(SPECIMENTESTTHRESHOLD.ReportTypeId))
					{
						if (!dr.IsNull(SPECIMENTESTTHRESHOLD.ReportTypeId))
						{
							if ((int)dr[SPECIMENTESTTHRESHOLD.ReportTypeId] < 1)
								dr[SPECIMENTESTTHRESHOLD.ReportTypeId] = DBNull.Value;
						}
					}

					if (dtInsert.Columns.Contains(SPECIMENTESTTHRESHOLD.ComponentClassId))
					{
						if (!dr.IsNull(SPECIMENTESTTHRESHOLD.ComponentClassId))
						{
							if ((int)dr[SPECIMENTESTTHRESHOLD.ComponentClassId] < 1)
								dr[SPECIMENTESTTHRESHOLD.ComponentClassId] = DBNull.Value;
						}
					}
				}
				_dtArrayList.Add(dtInsert);
				_sprocArrayList.Add(Common.VbecsStoredProcs.InsertSpecimenTestThreshold.StoredProcName);
			}

			//
			DAL.LabTest.SaveLabTests(_sprocArrayList, _dtArrayList);
		}

		///<Developers>
		///	<Developer>Mike Smith</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/27/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="991"> 
		///		<ExpectedInput>Lab test object.</ExpectedInput>
		///		<ExpectedOutput>Lab test object with all it's properties reset.</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="355"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// Resets the  properties of lab test class to initial values.
		///</summary>
		public void Clear()
		{
			SpecimenTestThresholdGuid = Guid.NewGuid();
			ComponentClassId = System.Int32.MinValue;
			LabTestName = string.Empty;
			SpecimenType = Common.SpecimenType.Unknown;
			Threshold = string.Empty;
			ReportType = Common.ReportType.Unknown;
			LabTestId = System.Int32.MinValue;
			RecordStatusCode = Common.RecordStatusCode.Unknown;
			IsDirty = false;
			IsNew = true;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/6/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="6321"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Datatable</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="6903"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Static function returning empty schema of specimen test threshold data table
		/// </summary>
		/// <returns>Data table schema</returns>
		public static DataTable GetSpecimenTestThresholdTableSchema()
		{
			DataTable dt = new DataTable();
			dt.Columns.Add(SPECIMENTESTTHRESHOLD.SpecimenTestThresholdGuid,typeof(Guid));
			dt.Columns.Add(SPECIMENTESTTHRESHOLD.ComponentClassId,typeof(int));
			dt.Columns.Add(SPECIMENTESTTHRESHOLD.LabTestId,typeof(string));
			dt.Columns.Add(SPECIMENTESTTHRESHOLD.VistaLaboratoryTestName,typeof(string));
			dt.Columns.Add(SPECIMENTESTTHRESHOLD.SpecimenTypeCode,typeof(string));
			dt.Columns.Add(Common.VbecsTables.SpecimenType.SpecimenTypeText,typeof(string));
			dt.Columns.Add(SPECIMENTESTTHRESHOLD.ThresholdResult,typeof(string));
			dt.Columns.Add(SPECIMENTESTTHRESHOLD.ReportTypeId,typeof(int));
			dt.Columns.Add(SPECIMENTESTTHRESHOLD.DivisionCode,typeof(string));
			dt.Columns.Add(SPECIMENTESTTHRESHOLD.RecordStatusCode,typeof(char));
			dt.Columns.Add(SPECIMENTESTTHRESHOLD.LastUpdateUser,typeof(string));
			dt.Columns.Add(SPECIMENTESTTHRESHOLD.RowVersion,typeof(byte[]));
			return dt;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/6/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6322"> 
		///		<ExpectedInput>LabTest</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6323"> 
		///		<ExpectedInput>Invalid LabTest</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Static function returning a data table of VistA Lab Tests matching the given search string
		/// </summary>
		/// <param name="labTestName">Partial Lab Test Name serach string</param>
		/// <returns>Data table containing VistALink search results</returns>
		public static DataTable GetVistALabTests(string labTestName)
		{
			//BR_33.01
			return gov.va.med.vbecs.DAL.VAL.LaboratoryTest.GetLaboratoryTestData( labTestName);
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6324"> 
		///		<ExpectedInput>Valid parameters</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6325"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Static function returning a data table of VistA Lab tests results matching the given search string
		/// </summary>
		/// <param name="startDate">Start date</param>
		/// <param name="endDate">End date</param>
		/// <param name="divisionCode">Division code</param>
		/// <param name="tests">Array list of Lab Test IENs</param>
		/// <param name="patients">Array list of Vista Patient IDs</param>
		/// <returns>Data table of verified lab test results</returns>
		public static DataTable GetVistALabTestResults(DateTime startDate, DateTime endDate, string divisionCode, ArrayList tests, ArrayList patients)
		{
			DataSet ds = gov.va.med.vbecs.DAL.VAL.LaboratoryTest.GetLaboratoryTestResults(startDate, endDate, divisionCode, tests, patients);
			//
			if (ds.Tables.Count!=1)
			{
				throw new BOL.BusinessObjectException(Common.StrRes.SysErrMsg.Common.NoTablesFoundException("for lab test criteria").ResString);
			}
			//
			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Carl Jensem</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3467"> 
		///		<ExpectedInput>ComponentClassId, LabTestId</ExpectedInput>
		///		<ExpectedOutput>DataTable of Tests</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3468"> 
		///		<ExpectedInput>Invald parameters</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetSpecimenTestThresholds
		/// </summary>
		/// <param name="componentClassID"></param>
		/// <param name="labTestID"></param>
		/// <returns></returns>
		public static DataTable GetSpecimenTestThresholds(int componentClassID, string labTestID)
		{
			return ( DAL.LabTest.GetSpecimenTestThresholds(componentClassID, labTestID) );
		}


		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/18/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3465"> 
		///		<ExpectedInput>ComponentClassId, PatientTests DataTable</ExpectedInput>
		///		<ExpectedOutput>Arraylist</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3466"> 
		///		<ExpectedInput>Invalid Data</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// DoTestsExceedThreshold
		/// </summary>
		/// <param name="componentClassID"></param>
		/// <param name="dtPatientTests"></param>
		/// <returns></returns>
		public static ArrayList DoTestsExceedThreshold(int componentClassID, DataTable dtPatientTests)
		{
			ArrayList failedTests = new ArrayList();
			//string testID;
			foreach (DataRow drPatientTest in dtPatientTests.Rows)
			{
				if ( drPatientTest.IsNull(Common.VbecsTables.PatientTest.LabTestId) || drPatientTest.IsNull(Common.VbecsTables.PatientTest.TestResult) )
				{
					continue;
				}
				else
				{
					string labTestID = drPatientTest[Common.VbecsTables.PatientTest.LabTestId].ToString();
					DataTable dtSpecimenThresholds = BOL.LabTest.GetSpecimenTestThresholds(componentClassID, labTestID);
					
					if (dtSpecimenThresholds != null && dtSpecimenThresholds.Rows.Count > 0)
					{
						DataRow drSpecimenThreshold = dtSpecimenThresholds.Rows[0];
					
						if ( !drSpecimenThreshold.IsNull(SPECIMENTESTTHRESHOLD.ThresholdResult) )
						{
							string threshold = drSpecimenThreshold[SPECIMENTESTTHRESHOLD.ThresholdResult].ToString();
							if ( (threshold.IndexOf("<") != -1  && threshold.IndexOf(">") != -1) || threshold.Length > 1 )
							{
								// get threshold value
								string arrow = threshold.Substring(0,1);
								decimal thresholdValue = Convert.ToDecimal(threshold.Substring(1));
								// get patient test value
								// This must be in a try/catch block because it cannot be guaranteed that TestResult will
								// actually be a numerical value.  For some tests, users are allowed to enter alphanumeric 
								// text.
								try
								{
									decimal result = Convert.ToDecimal( drPatientTest[Common.VbecsTables.PatientTest.TestResult].ToString() );
							
									// compare!
									if (arrow == "<")
									{
										if (result < thresholdValue)
										{
											failedTests.Add( drSpecimenThreshold[SPECIMENTESTTHRESHOLD.VistaLaboratoryTestName].ToString() );
										}
									}
									else
									{
										if (result > thresholdValue)
										{
											failedTests.Add( drSpecimenThreshold[SPECIMENTESTTHRESHOLD.VistaLaboratoryTestName].ToString() );
										}
									}
								}
								catch (System.FormatException)
								{
									// Do nothing: this will result if a non-numeric value was entered as a test result.
									// Since there is no way to compare such a value, we'll ignore it.
								}
							}
						}
					}
				}
			}
			return failedTests;
		}

		#endregion 
	}
}


